<?php

namespace App\Http\Controllers\Install;

use App\Http\Controllers\Controller;
use App\Utils\ModuleUtil;
use Illuminate\Http\Request;
use Module;
use ZipArchive;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Permission;
use Spatie\Permission\Models\Role;

class ModulesController extends Controller
{
    protected $moduleUtil;

    public function __construct(ModuleUtil $moduleUtil)
    {
        $this->moduleUtil = $moduleUtil;
    }

    public function index()
    {
        if (! auth()->user()->can('manage_modules')) {
            abort(403, 'Unauthorized action.');
        }

        $notAllowed = $this->moduleUtil->notAllowedInDemo();
        if (! empty($notAllowed)) {
            return $notAllowed;
        }

        $modules = Module::toCollection()->toArray();

        foreach ($modules as $module => $details) {
            $modules[$module]['is_installed'] = $this->moduleUtil->isModuleInstalled($details['name']);

            if ($modules[$module]['is_installed']) {
                $modules[$module]['version'] = $this->moduleUtil->getModuleVersionInfo($details['name']);
            }

            try {
                $modules[$module]['install_link'] = action('\\Modules\\'.$details['name'].'\\Http\\Controllers\\InstallController@index');
            } catch (\Exception $e) {
                $modules[$module]['install_link'] = '#';
            }

            try {
                $modules[$module]['update_link'] = action('\\Modules\\'.$details['name'].'\\Http\\Controllers\\InstallController@update');
            } catch (\Exception $e) {
                $modules[$module]['update_link'] = '#';
            }

            try {
                $modules[$module]['uninstall_link'] = action('\\Modules\\'.$details['name'].'\\Http\\Controllers\\InstallController@uninstall');
            } catch (\Exception $e) {
                $modules[$module]['uninstall_link'] = '#';
            }
        }

        $is_demo = (config('app.env') == 'demo');
        $mods = $this->__available_modules();

        return view('install.modules.index')
            ->with(compact('modules', 'is_demo', 'mods'));
    }

    public function regenerate()
    {
        if (! auth()->user()->can('manage_modules')) {
            abort(403, 'Unauthorized action.');
        }

        $notAllowed = $this->moduleUtil->notAllowedInDemo();
        if (! empty($notAllowed)) {
            return $notAllowed;
        }

        try {
            Artisan::call('module:publish');
            Artisan::call('passport:install --force');

            $output = ['success' => 1, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            $output = ['success' => 0, 'msg' => $e->getMessage()];
        }

        return redirect()->back()->with('status', $output);
    }

    public function update(Request $request, $module_name)
    {
        if (! auth()->user()->can('manage_modules')) {
            abort(403, 'Unauthorized action.');
        }

        $notAllowed = $this->moduleUtil->notAllowedInDemo();
        if (! empty($notAllowed)) {
            return $notAllowed;
        }

        try {
            $module = Module::find($module_name);

            if ($request->action_type == 'activate') {
                $module->enable();
            } elseif ($request->action_type == 'deactivate') {
                $module->disable();
            }

            $output = ['success' => true, 'msg' => __('lang_v1.success')];
        } catch (\Exception $e) {
            $output = ['success' => false, 'msg' => $e->getMessage()];
        }

        return redirect()->back()->with(['status' => $output]);
    }

    public function destroy($module_name)
    {
        if (! auth()->user()->can('manage_modules')) {
            abort(403, 'Unauthorized action.');
        }

        $notAllowed = $this->moduleUtil->notAllowedInDemo();
        if (! empty($notAllowed)) {
            return $notAllowed;
        }

        try {
            $module = Module::find($module_name);
            $path = $module->getPath();
            die("To delete the module delete this folder <br/>" . $path . '<br/> Go back after deleting');
        } catch (\Exception $e) {
            $output = ['success' => false, 'msg' => $e->getMessage()];
            return redirect()->back()->with(['status' => $output]);
        }
    }

    public function uploadModule(Request $request)
    {
        $notAllowed = $this->moduleUtil->notAllowedInDemo();
        if (! empty($notAllowed)) {
            return $notAllowed;
        }

        try {
            $module = $request->file('module');
            $module_name = str_replace('.zip', '', $module->getClientOriginalName());

            if ($module->getMimeType() != 'application/zip') {
                return redirect()->back()->with(['status' => [
                    'success' => false,
                    'msg' => __('lang_v1.pls_upload_valid_zip_file')
                ]]);
            }

            $path = '../Modules';
            if (! is_dir($path)) {
                mkdir($path, 0777, true);
            }

            $zip = new ZipArchive();
            if ($zip->open($module) === true) {
                $zip->extractTo($path.'/');
                $zip->close();

                // Auto-register permissions for M-PESA and attach to Superadmin
                if ($module_name === 'MpesaIntegration') {
                    $permissions = [
                        'mpesa.view',
                        'mpesa.receive',
                        'mpesa.reverse',
                        'mpesa.transactions',
                        'mpesa.settings',
                        'manage_modules'
                    ];

                    foreach ($permissions as $perm) {
                        Permission::firstOrCreate(['name' => $perm, 'guard_name' => 'web']);
                    }

                    $superadmin = Role::where('name', 'Superadmin')->first();
                    if ($superadmin) {
                        $superadmin->givePermissionTo($permissions);
                    }
                }
            }

            return redirect()->back()->with(['status' => [
                'success' => true,
                'msg' => __('lang_v1.success')
            ]]);
        } catch (\Exception $e) {
            return redirect()->back()->with(['status' => [
                'success' => false,
                'msg' => __('messages.something_went_wrong')
            ]]);
        }
    }

    private function __available_modules()
    {
        $modules = [];

        $modules[] = (object)['n' => 'Essentials', 'dn' => 'Essentials Module', 'u' => 'https://ultimatefosters.com/recommends/essential-app/', 'd' => 'Essentials features for every growing businesses.'];
        $modules[] = (object)['n' => 'Superadmin', 'dn' => 'Superadmin Module', 'u' => 'https://ultimatefosters.com/recommends/superadmin-app/', 'd' => 'Turn your POS to SaaS application and start earning by selling subscriptions'];
        $modules[] = (object)['n' => 'Woocommerce', 'dn' => 'Woocommerce Module', 'u' => 'https://ultimatefosters.com/recommends/woocommerce-app/', 'd' => 'Sync your Woocommerce store with POS'];
        $modules[] = (object)['n' => 'Manufacturing', 'dn' => 'Manufacturing Module', 'u' => 'https://ultimatefosters.com/recommends/manufacturing-app/', 'd' => 'Manufacture products from raw materials, organise recipe & ingredients'];
        $modules[] = (object)['n' => 'Project', 'dn' => 'Project Module', 'u' => 'https://ultimatefosters.com/recommends/project-app/', 'd' => 'Manage Projects, tasks, tasks time logs, activities and much more.'];
        $modules[] = (object)['n' => 'Repair', 'dn' => 'Repair Module', 'u' => 'https://ultimatefosters.com/recommends/repair-app/', 'd' => 'Repair module helps with complete repair service management of electronic goods.'];
        $modules[] = (object)['n' => 'Crm', 'dn' => 'CRM Module', 'u' => 'https://ultimatefosters.com/product/crm-module-for-ultimatepos/', 'd' => 'Customer relationship management module'];
        $modules[] = (object)['n' => 'ProductCatalogue', 'dn' => 'ProductCatalogue', 'u' => 'https://codecanyon.net/item/digital-product-catalogue-menu-module-for-ultimatepos/28825346', 'd' => 'Digital Product catalogue Module'];
        $modules[] = (object)['n' => 'Accounting', 'dn' => 'Accounting Module', 'u' => 'https://ultimatefosters.com/product/accounting-bookkeeping-module-for-ultimatepos/', 'd' => 'Accounting & Book keeping module for UltimatePOS'];
        $modules[] = (object)['n' => 'AiAssistance', 'dn' => 'AiAssistance Module', 'u' => 'https://ultimatefosters.com/product/ai-assistance-module-for-ultimatepos/', 'd' => 'AI Assistant module for UltimatePOS. Uses OpenAI API for copywriting & reporting'];
        $modules[] = (object)['n' => 'AssetManagement', 'dn' => 'AssetManagement Module', 'u' => 'https://ultimatefosters.com/product/asset-management-module-for-ultimatepos/', 'd' => 'Useful for managing all kinds of assets.'];
        $modules[] = (object)['n' => 'Cms', 'dn' => 'Cms Module', 'u' => 'https://ultimatefosters.com/product/ultimatepos-cms-module/', 'd' => 'Mini CMS to manage landing pages, blogs, contact us & other frontend content'];
        $modules[] = (object)['n' => 'Connector', 'dn' => 'Connector/API Module', 'u' => 'https://ultimatefosters.com/product/rest-api-module-for-ultimatepos/', 'd' => 'Provide the API for POS.'];
        $modules[] = (object)['n' => 'Gym', 'dn' => 'Gym Module', 'u' => 'https://ultimatefosters.com/product/gym-management-module-for-ultimatepos/', 'd' => 'Gym Management module for UltimatePOS'];
        $modules[] = (object)['n' => 'Hms', 'dn' => 'Hotel Management Module', 'u' => 'https://ultimatefosters.com/product/hms-hotel-management-system-module-for-ultimatepos/', 'd' => 'Hotel Management System module for UltimatePOS.'];
        $modules[] = (object)['n' => 'MpesaIntegration', 'dn' => 'M-PESA Integration Module', 'u' => 'https://navontech.co.ke/modules/mpesa', 'd' => 'Accept M-PESA STK Push, C2B and B2C payments with tenant-specific credentials.'];

        return serialize($modules);
    }
}
