<?php

namespace Modules\Superadmin\Http\Controllers;

use App\System;
use App\Utils\BusinessUtil;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Routing\Controller;
use App\Models\MpesaCredential;

class SuperadminSettingsController extends Controller
{
    protected $businessUtil;
    protected $mailDrivers;
    protected $backupDisk;

    public function __construct(BusinessUtil $businessUtil)
    {
        $this->businessUtil = $businessUtil;

        $this->mailDrivers = [
            'smtp' => 'SMTP',
            'sendmail' => 'Sendmail',
            'mailgun' => 'Mailgun',
            'mandrill' => 'Mandrill',
            'ses' => 'SES',
            'sparkpost' => 'Sparkpost',
        ];

        $this->backupDisk = ['local' => 'Local', 'dropbox' => 'Dropbox'];
    }

    public function edit()
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        $settings = System::pluck('value', 'key');
        $currencies = $this->businessUtil->allCurrencies();
        $superadmin_version = System::getProperty('superadmin_version');
        $is_demo = env('APP_ENV') == 'demo';

        $default_values = [
            'APP_NAME' => env('APP_NAME'),
            'APP_TITLE' => env('APP_TITLE'),
            'APP_LOCALE' => env('APP_LOCALE'),
            'MAIL_MAILER' => $is_demo ? null : env('MAIL_MAILER'),
            'MAIL_HOST' => $is_demo ? null : env('MAIL_HOST'),
            'MAIL_PORT' => $is_demo ? null : env('MAIL_PORT'),
            'MAIL_USERNAME' => $is_demo ? null : env('MAIL_USERNAME'),
            'MAIL_PASSWORD' => $is_demo ? null : env('MAIL_PASSWORD'),
            'MAIL_ENCRYPTION' => $is_demo ? null : env('MAIL_ENCRYPTION'),
            'MAIL_FROM_ADDRESS' => $is_demo ? null : env('MAIL_FROM_ADDRESS'),
            'MAIL_FROM_NAME' => $is_demo ? null : env('MAIL_FROM_NAME'),
            'STRIPE_PUB_KEY' => $is_demo ? null : env('STRIPE_PUB_KEY'),
            'STRIPE_SECRET_KEY' => $is_demo ? null : env('STRIPE_SECRET_KEY'),
            'PAYPAL_MODE' => env('PAYPAL_MODE'),
            'PAYPAL_CLIENT_ID' => $is_demo ? null : env('PAYPAL_CLIENT_ID'),
            'PAYPAL_APP_SECRET' => $is_demo ? null : env('PAYPAL_APP_SECRET'),
            'BACKUP_DISK' => env('BACKUP_DISK'),
            'DROPBOX_ACCESS_TOKEN' => $is_demo ? null : env('DROPBOX_ACCESS_TOKEN'),
            'RAZORPAY_KEY_ID' => $is_demo ? null : env('RAZORPAY_KEY_ID'),
            'RAZORPAY_KEY_SECRET' => $is_demo ? null : env('RAZORPAY_KEY_SECRET'),
            'PESAPAL_CONSUMER_KEY' => $is_demo ? null : env('PESAPAL_CONSUMER_KEY'),
            'PESAPAL_CONSUMER_SECRET' => $is_demo ? null : env('PESAPAL_CONSUMER_SECRET'),
            'PESAPAL_LIVE' => $is_demo ? null : (env('PESAPAL_LIVE') ? 'true' : 'false'),
            'PUSHER_APP_ID' => $is_demo ? null : env('PUSHER_APP_ID'),
            'PUSHER_APP_KEY' => $is_demo ? null : env('PUSHER_APP_KEY'),
            'PUSHER_APP_SECRET' => $is_demo ? null : env('PUSHER_APP_SECRET'),
            'PUSHER_APP_CLUSTER' => $is_demo ? null : env('PUSHER_APP_CLUSTER'),
            'GOOGLE_MAP_API_KEY' => $is_demo ? null : env('GOOGLE_MAP_API_KEY'),
            'ALLOW_REGISTRATION' => $is_demo ? null : env('ALLOW_REGISTRATION'),
            'PAYSTACK_PUBLIC_KEY' => $is_demo ? null : env('PAYSTACK_PUBLIC_KEY'),
            'PAYSTACK_SECRET_KEY' => $is_demo ? null : env('PAYSTACK_SECRET_KEY'),
            'FLUTTERWAVE_PUBLIC_KEY' => $is_demo ? null : env('FLUTTERWAVE_PUBLIC_KEY'),
            'FLUTTERWAVE_SECRET_KEY' => $is_demo ? null : env('FLUTTERWAVE_SECRET_KEY'),
            'FLUTTERWAVE_ENCRYPTION_KEY' => $is_demo ? null : env('FLUTTERWAVE_ENCRYPTION_KEY'),
            'MY_FATOORAH_API_KEY' => $is_demo ? null : env('MY_FATOORAH_API_KEY'),
            'MY_FATOORAH_COUNTRY_ISO' => $is_demo ? null : env('MY_FATOORAH_COUNTRY_ISO'),
            'MY_FATOORAH_IS_TEST' => $is_demo ? null : (env('MY_FATOORAH_IS_TEST') ? 'true' : 'false'),
        ];

        // Fetch M-PESA credentials for current business
        $mpesa_credentials = MpesaCredential::where('business_id', auth()->user()->business_id)->first();
        if ($mpesa_credentials) {
            $default_values['MPESA_SHORTCODE'] = $mpesa_credentials->shortcode;
            $default_values['MPESA_CONSUMER_KEY'] = $mpesa_credentials->consumer_key;
            $default_values['MPESA_CONSUMER_SECRET'] = $mpesa_credentials->consumer_secret;
            $default_values['MPESA_PASSKEY'] = $mpesa_credentials->passkey;
            $default_values['MPESA_VALIDATION_URL'] = $mpesa_credentials->validation_url;
            $default_values['MPESA_CONFIRMATION_URL'] = $mpesa_credentials->confirmation_url;
            $default_values['MPESA_ENVIRONMENT'] = $mpesa_credentials->environment;
        }

        $mail_drivers = $this->mailDrivers;

        $config_languages = config('constants.langs');
        $languages = [];
        foreach ($config_languages as $key => $value) {
            $languages[$key] = $value['full_name'];
        }

        $backup_disk = $this->backupDisk;
        $cron_job_command = $this->businessUtil->getCronJobCommand();

        return view('superadmin::superadmin_settings.edit')
            ->with(compact(
                'currencies',
                'settings',
                'superadmin_version',
                'mail_drivers',
                'languages',
                'default_values',
                'backup_disk',
                'cron_job_command'
            ));
    }

    public function update(Request $request)
    {
        if (!auth()->user()->can('superadmin')) {
            abort(403, 'Unauthorized action.');
        }

        try {
            if (config('app.env') == 'demo') {
                return back()->with('status', ['success' => 0, 'msg' => 'Feature disabled in demo!!']);
            }

            $system_settings = $request->only([
                'app_currency_id', 'invoice_business_name', 'email',
                'invoice_business_landmark', 'invoice_business_zip',
                'invoice_business_state', 'invoice_business_city', 'invoice_business_country',
                'package_expiry_alert_days', 'superadmin_register_tc',
                'welcome_email_subject', 'welcome_email_body',
                'additional_js', 'additional_css', 'offline_payment_details'
            ]);

            $checkboxes = [
                'enable_business_based_username',
                'superadmin_enable_register_tc',
                'allow_email_settings_to_businesses',
                'enable_new_business_registration_notification',
                'enable_new_subscription_notification',
                'enable_welcome_email',
                'enable_offline_payment'
            ];

            foreach ($checkboxes as $checkbox) {
                $system_settings[$checkbox] = !empty($request->input($checkbox)) ? 1 : 0;
            }

            foreach ($system_settings as $key => $setting) {
                System::updateOrCreate(['key' => $key], ['value' => $setting]);
            }

            $env_settings = $request->only([
                'APP_NAME', 'APP_TITLE', 'APP_LOCALE',
                'MAIL_MAILER', 'MAIL_HOST', 'MAIL_PORT',
                'MAIL_USERNAME', 'MAIL_PASSWORD', 'MAIL_ENCRYPTION',
                'MAIL_FROM_ADDRESS', 'MAIL_FROM_NAME',
                'STRIPE_PUB_KEY', 'STRIPE_SECRET_KEY',
                'PAYPAL_MODE', 'PAYPAL_CLIENT_ID', 'PAYPAL_APP_SECRET',
                'BACKUP_DISK', 'DROPBOX_ACCESS_TOKEN',
                'RAZORPAY_KEY_ID', 'RAZORPAY_KEY_SECRET',
                'PESAPAL_CONSUMER_KEY', 'PESAPAL_CONSUMER_SECRET', 'PESAPAL_LIVE',
                'PUSHER_APP_ID', 'PUSHER_APP_KEY', 'PUSHER_APP_SECRET', 'PUSHER_APP_CLUSTER',
                'GOOGLE_MAP_API_KEY', 'PAYSTACK_SECRET_KEY', 'PAYSTACK_PUBLIC_KEY',
                'FLUTTERWAVE_PUBLIC_KEY', 'FLUTTERWAVE_SECRET_KEY', 'FLUTTERWAVE_ENCRYPTION_KEY',
                'MAPBOX_ACCESS_TOKEN', 'MY_FATOORAH_API_KEY', 'MY_FATOORAH_IS_TEST', 'MY_FATOORAH_COUNTRY_ISO'
            ]);

            $env_settings['ALLOW_REGISTRATION'] = !empty($request->input('ALLOW_REGISTRATION')) ? 'true' : 'false';
            $env_settings['BROADCAST_DRIVER'] = 'pusher';

            $env_path = base_path('.env');
            $env_lines = file($env_path);
            $found_envs = [];

            foreach ($env_settings as $key => $value) {
                foreach ($env_lines as $i => $line) {
                    if (strpos($line, $key) !== false) {
                        $env_lines[$i] = $key . '="' . $value . '"' . PHP_EOL;
                        $found_envs[] = $key;
                    }
                }
            }

            $missing_envs = array_diff(array_keys($env_settings), $found_envs);
            foreach ($missing_envs as $k => $key) {
                $env_lines[] = ($k == 0 ? PHP_EOL : '') . $key . '="' . $env_settings[$key] . '"' . PHP_EOL;
            }

            if (is_writable($env_path) && file_put_contents($env_path, implode('', $env_lines))) {
                $output = ['success' => 1, 'msg' => __('lang_v1.success')];
            } else {
                $output = ['success' => 0, 'msg' => 'Could not save .env changes. Ensure proper file permissions.'];
            }
        } catch (\Exception $e) {
            \Log::emergency('File:' . $e->getFile() . ' Line:' . $e->getLine() . ' Message:' . $e->getMessage());
            $output = ['success' => 0, 'msg' => __('messages.something_went_wrong')];
        }

        // Save M-PESA credentials
        MpesaCredential::updateOrCreate(
            ['business_id' => auth()->user()->business_id],
            [
                'shortcode' => $request->input('MPESA_SHORTCODE'),
                'consumer_key' => $request->input('MPESA_CONSUMER_KEY'),
                'consumer_secret' => $request->input('MPESA_CONSUMER_SECRET'),
                'passkey' => $request->input('MPESA_PASSKEY'),
                'validation_url' => $request->input('MPESA_VALIDATION_URL'),
                'confirmation_url' => $request->input('MPESA_CONFIRMATION_URL'),
                'environment' => $request->input('MPESA_ENVIRONMENT'),
            ]
        );

        return redirect()
            ->action([self::class, 'edit'])
            ->with('status', $output);
    }
}
