<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use App\Models\MpesaCredential;

class MpesaController extends Controller
{
    /**
     * Trigger M-PESA STK Push.
     */
    public function stkPush(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'amount' => 'required|numeric|min:1',
            'phone' => 'required|regex:/^\+?\d{10,13}$/',
        ]);

        if ($validator->fails()) {
            return response()->json(['success' => false, 'message' => 'Invalid request data', 'errors' => $validator->errors()], 422);
        }

        $business_id = auth()->user()->business_id;
        $mpesa = MpesaCredential::where('business_id', $business_id)->first();

        if (!$mpesa) {
            return response()->json(['success' => false, 'message' => 'M-PESA credentials not found for this business.'], 404);
        }

        // Get access token
        $token_url = 'https://api.safaricom.co.ke/oauth/v1/generate?grant_type=client_credentials';
        $credentials = base64_encode($mpesa->consumer_key . ":" . $mpesa->consumer_secret);

        $token_response = Http::withHeaders([
            'Authorization' => 'Basic ' . $credentials,
        ])->get($token_url);

        if (!$token_response->ok()) {
            return response()->json(['success' => false, 'message' => 'Failed to get access token.'], 500);
        }

        $access_token = $token_response['access_token'];

        // Prepare STK push request
        $stk_url = 'https://api.safaricom.co.ke/mpesa/stkpush/v1/processrequest';
        $timestamp = now()->format('YmdHis');
        $password = base64_encode($mpesa->shortcode . $mpesa->passkey . $timestamp);

        $stk_payload = [
            "BusinessShortCode" => $mpesa->shortcode,
            "Password" => $password,
            "Timestamp" => $timestamp,
            "TransactionType" => "CustomerPayBillOnline",
            "Amount" => $request->amount,
            "PartyA" => $request->phone,
            "PartyB" => $mpesa->shortcode,
            "PhoneNumber" => $request->phone,
            "CallBackURL" => route('mpesa.callback'),
            "AccountReference" => "POS Payment",
            "TransactionDesc" => "POS Sale Payment"
        ];

        $stk_response = Http::withToken($access_token)
            ->withHeaders(['Content-Type' => 'application/json'])
            ->post($stk_url, $stk_payload);

        if (!$stk_response->ok()) {
            Log::error('M-PESA STK Push failed:', $stk_response->json());
            return response()->json(['success' => false, 'message' => 'STK Push request failed.'], 500);
        }

        return response()->json(['success' => true, 'message' => 'STK Push request sent.', 'data' => $stk_response->json()]);
    }

    /**
     * Handle M-PESA STK Callback.
     */
    public function handleCallback(Request $request)
    {
        $callback_data = $request->all();

        // Log callback for now (store to DB later if needed)
        Log::info('M-PESA Callback received:', $callback_data);

        // You can expand this to store transactions in a table, mark payment complete, etc.

        return response()->json(['ResultCode' => 0, 'ResultDesc' => 'Accepted']);
    }
}
